/*
*  Copyright 2019-2020 Zheng Jie
*
*  Licensed under the Apache License, Version 2.0 (the "License");
*  you may not use this file except in compliance with the License.
*  You may obtain a copy of the License at
*
*  http://www.apache.org/licenses/LICENSE-2.0
*
*  Unless required by applicable law or agreed to in writing, software
*  distributed under the License is distributed on an "AS IS" BASIS,
*  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
*  See the License for the specific language governing permissions and
*  limitations under the License.
*/
package com.zthzinfo.contract.controller;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.conditions.update.LambdaUpdateWrapper;
import com.zthzinfo.aliyun.oss.ExtUtil;
import com.zthzinfo.aliyun.oss.OSSObject;
import com.zthzinfo.aliyun.oss.ResultFile;
import com.zthzinfo.common.ResponseMapBuilder;
import com.zthzinfo.contract.domain.ContractChangeStatusLog;
import com.zthzinfo.contract.domain.CtrtContract;
import com.zthzinfo.contract.domain.TplContract;
import com.zthzinfo.contract.enums.*;
import com.zthzinfo.contract.interfaceapi.IContractDataProvider;
import com.zthzinfo.contract.interfaceapi.vo.CtrtCompanyDto;
import com.zthzinfo.contract.interfaceapi.vo.CtrtContactDto;
import com.zthzinfo.contract.interfaceapi.vo.CtrtPageDto;
import com.zthzinfo.contract.model.dto.ContractDTO;
import com.zthzinfo.contract.model.dto.TplContractDTO;
import com.zthzinfo.contract.service.*;
import com.zthzinfo.contract.service.dto.CtrtContractQueryCriteria;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Pageable;
import org.springframework.data.web.PageableDefault;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

/**
* @author sunzsh
* @date 2023-06-07 10:13:59
**/
@Slf4j
@RestController("ctrtContractController")
@RequiredArgsConstructor
@Api(tags = "合同管理")
@RequestMapping("/ctrt-service")
public class ContractController {
    public static final String VIEW_PRE = "ctrt-view";

    private final CtrtContractService ctrtContractService;
    private final CtrtContractClausesService clausesService;
    private final CtrtTplContractClausesService ctrtTplContractClausesService;
    private final CtrtTplContractService ctrtTplContractService;
    private final CtrtContractChangeStatusLogService contractChangeStatusLogService;
    private final CtrtIdGenService ctrtIdGenService;

    @Autowired(required = false)
    protected OSSObject ossObject;

    @Autowired(required = false)
    IContractDataProvider dataProvider;


    /**
     * 跳转到创建合同页面
     */
    @RequestMapping("/view/create")
    public ModelAndView contract(@RequestParam(name = "tplModules", required = false) List<Integer> tplModules) {
        // 根据版块获取合同模板
        Map<Integer,List<TplContractDTO>> templates = getContractTplByModule(tplModules);

        ModelAndView mv = new ModelAndView();
        mv.addObject("templates", templates);
        mv.addObject("step", 0);

        mv.setViewName(VIEW_PRE + "/index");
        return mv;
    }

    /**
     * 跳转到修改合同页面
     */
    @RequestMapping("/view/modify")
    public ModelAndView modify(@RequestParam(name = "tplModules", required = false) List<Integer> tplModules,
                               @RequestParam("contractId") String contractId) {
        // 根据版块获取合同模板
        Map<Integer,List<TplContractDTO>> templates = getContractTplByModule(tplModules);


        // 查询某个合同展示详情
        ContractDTO contract = ctrtContractService.selectById(contractId);


        ModelAndView mv = new ModelAndView();

        Assert.notNull(contract, "系统中未找到该合同");
        mv.addObject("contractId", contractId);
        mv.addObject("entity", contract);

        mv.addObject("templates", templates);
        mv.addObject("step", 0);

        mv.setViewName(VIEW_PRE + "/index");
        return mv;
    }

    /**
     * 根据公司id查询联系人
     */
    @RequestMapping("/getContactByCompanyId")
    public ResponseEntity<Object> getContactByCompanyId(
        @RequestParam(value = "companyIdA",required = false) String companyIdA,
        @RequestParam(value = "companyIdB",required = false) String companyIdB,
        @RequestParam("currentRole") String currentRole
    ) {
        if (dataProvider == null) {
            return new ResponseEntity<>(new Object(), HttpStatus.OK);
        }
        CtrtContactDto[] contactByCompanyId = dataProvider.getContactByCompanyId(companyIdA, companyIdB, currentRole);
        Map<String, CtrtContactDto> contactInMap = new HashMap<>();
        contactInMap.put("companyA", contactByCompanyId.length >= 1 ? contactByCompanyId[0] : null);
        contactInMap.put("companyB", contactByCompanyId.length >= 2 ? contactByCompanyId[1] : null);


        return new ResponseEntity<>(contactInMap, HttpStatus.OK);
    }

    /**
     * 根据公司id查询联系人列表让用户选择
     */
    @RequestMapping("/contactsByContactId")
    public ResponseEntity<Object> contactsByContactId(@RequestParam("customerId") String customerId,
                        @RequestParam("searchForRole") String searchForRole,
                        @RequestParam("currentRole") String currentRole) {
        if (dataProvider == null) {
            return new ResponseEntity<>(new Object(), HttpStatus.OK);
        }
        List<CtrtContactDto> contactByCustomerId = dataProvider.contactByCustomerId(customerId, searchForRole, currentRole);
        return new ResponseEntity<>(contactByCustomerId, HttpStatus.OK);
    }

    /**
     * 罗列公司列表让用户选择
     */
    @RequestMapping("/partyChooseList")
    public ResponseEntity<Object> partyChooseList(@RequestParam("page") Integer page,
                                                  @RequestParam("size") Integer size,
                                                  @RequestParam(name = "keyword", required = false) String keyword,
                                                  @RequestParam("searchForRole") String searchForRole,
                                                  @RequestParam("currentRole") String currentRole) {
        if (dataProvider == null) {
            return new ResponseEntity<>(new ArrayList<>(), HttpStatus.OK);
        }
        CtrtPageDto<CtrtCompanyDto> pageDto = dataProvider.listCompanyForChoose(page, size, keyword, searchForRole, currentRole);
        return new ResponseEntity<>(
            pageDto,
            HttpStatus.OK
        );
    }
    /**
     * 根据客户id罗列账户让用户选择
     */
    @RequestMapping("/accountList")
    public ResponseEntity<Object> accountList(@RequestParam("id") String id,
                                              @RequestParam("searchForRole") String searchForRole,
                                              @RequestParam("currentRole") String currentRole) {
        if (dataProvider == null) {
            return new ResponseEntity<>(new ArrayList<>(), HttpStatus.OK);
        }
        return new ResponseEntity<>(
            dataProvider.accountList(id, searchForRole, currentRole),
            HttpStatus.OK
        );
    }

    /** 更新签订日期 */
    @PostMapping("/update_signdate")
    public ResponseEntity<Object> updateSignDate(@RequestBody ContractDTO res) {
        String contractNo = ctrtContractService.updateSignDate(res);
        return new ResponseEntity<>(contractNo, HttpStatus.OK);
    }

//    /**
//     * 加载当前登录人所在公司ID
//     */
//    private void loadCurrentCompanyId(ModelAndView mv,String userId) {
//        List<UserCompany> companies = wxDepartmentService.getUserPlateInfoByUserId(userId, 1);
//        if(CollUtil.isEmpty(companies)) {
//            log.warn("当前登录人 {},不属于任何一个公司,请检查该用户是否配置了公司",SecurityUtils.getCurrentUserId());
//            return;
//        }
//        companies.stream()
//                .map(UserCompany::getCFullId)
//                .filter(StrUtil::isNotBlank)
//                .findFirst()
//                .ifPresent(companyId -> {
//                    mv.addObject("currentCompanyId", companyId);
//                });
//    }

//    /**
//     * 配置内部公司的字典ID
//     */
//    private void loadInnerCompanyDictId(ModelAndView mv) {
//        DictDto dict = dictService.getByKey(AppConstants.KEY_INNER_COMPANY_ID);
//        if (dict == null) {
//            log.warn("未配置内部公司的字典ID,请配置{}",AppConstants.KEY_INNER_COMPANY_ID);
//            return;
//        }
//        mv.addObject("innerCompanyDictId",dict.getDescription());
//    }

    private ModelAndView handleContractInfo(String contractId) {
        CtrtContract contract = ctrtContractService.getById(contractId);
        Assert.notNull(contract, "系统中未找到该合同");
        ContractDTO dto = new ContractDTO();
        BeanUtils.copyProperties(contract,dto);
        dto.setContractTypeName(ContractTypeEnums.parseName(dto.getContractType()));
        dto.setSpStatusName(ProcessStatusEnum.getDescByKey(dto.getSpStatus()));

        ModelAndView mv = new ModelAndView();
        mv.addObject("info", dto);
        mv.setViewName(VIEW_PRE + "info");

        return mv;
    }

    /**
     * 板块处理
     */
    private List<Integer> moduleHandle(String source, Integer tplModule) {
        List<Integer> tplModules = CollUtil.newArrayList();
        if(tplModule == null) {
            if (StrUtil.equals(source, ContractSourceEnums.GANGCAI.getValue())) {
                // 如果是钢材物流系统过来的，合同类型就是钢材
                tplModules.add(ContractModuleEnums.GANGCAI.getKey());
                tplModules.add(ContractModuleEnums.WULIU_OTHER.getKey());
            } else {
                // 根据当前登录人确定板块
//                List<UserCompany> plates = wxDepartmentService.getUserPlateInfoByUserId(user.getUserId(), 0);
//                if (CollUtil.isNotEmpty(plates)) {
//                    DictDto wuliuPlate = dictService.getByKey(AppConstants.KEY_WULIU_PLATE_ID);
//                    if (wuliuPlate == null) {
//                        log.error("请配置字典：key=[{}],desc=物理板块字典ID ",AppConstants.KEY_WULIU_PLATE_ID);
//                    }
//                    DictDto gangcaiDict = dictService.getByKey(AppConstants.KEY_GANGCAI_WXDEPTIDS);
//                    List<String> gangcaiIds = Optional.ofNullable(gangcaiDict).map(DictDto::getDictDetails).orElse(new ArrayList<>())
//                            .stream().map(DictDetailDto::getValue)
//                            .distinct().collect(Collectors.toList());
//
//                    DictDto sanhuoDict = dictService.getByKey(AppConstants.KEY_SANHUO_WXDEPTIDS);
//                    List<String> sanhuoIds = Optional.ofNullable(sanhuoDict).map(DictDto::getDictDetails).orElse(new ArrayList<>())
//                            .stream().map(DictDetailDto::getValue)
//                            .distinct().collect(Collectors.toList());
//
//                    plates.stream().filter(item -> StrUtil.equals(item.getPlateId(), wuliuPlate.getDescription()))
//                            .forEach(item -> {
//                                if (gangcaiIds.contains(item.getCid())) {
//                                    // 钢材
//                                    if(tplModules.contains(ContractModuleEnums.GANGCAI.getKey())) {
//                                        tplModules.add(ContractModuleEnums.GANGCAI.getKey());
//                                        tplModules.add(ContractModuleEnums.WULIU_OTHER.getKey());
//                                    }
//                                } else if (sanhuoIds.contains(item.getCid())) {
//                                    // 散货
//                                    if(tplModules.contains(ContractModuleEnums.SANHUO.getKey())) {
//                                        tplModules.add(ContractModuleEnums.SANHUO.getKey());
//                                        tplModules.add(ContractModuleEnums.WULIU_OTHER.getKey());
//                                    }
//                                }
//                            });
//                }
                // 添加其他
                tplModules.add(ContractModuleEnums.OTHER.getKey());
            }
        } else {
            tplModules.add(tplModule);
            if (StrUtil.equals(source, ContractSourceEnums.GANGCAI.getValue())) {
                // 如果是钢材物流系统过来的，合同类型就是钢材
                tplModules.add(ContractModuleEnums.WULIU_OTHER.getKey());
            }
        }
        return tplModules;
    }

    /**
     * 生成合同号
     */
    @GetMapping("/genNo")
    public ResponseEntity<Object> genNo(
            String companyCode,
            Integer contractType
    ) {
//        companyCode = Optional.ofNullable(companyCode).orElse(AppConstants.DEFAULT_COMPANY_CODE);
//        String deptCode = Optional.ofNullable(wxDepartmentService.getCurrentUserDeptCode()).orElse(AppConstants.DEFAULT_DEPT_CODE);
        String number = ctrtIdGenService.genNumber(companyCode, "TEST", ContractTypeEnums.parse(contractType).getCode());
        return new ResponseEntity<>(number,HttpStatus.OK);
    }

    /**
     * 客户
     */
//    @AnonymousAccess
    @GetMapping("/customer/list")
    public ResponseEntity<Object> getCustomerList(@PageableDefault Pageable pageable) {
        return new ResponseEntity<>(new ArrayList<>(), HttpStatus.OK);
    }
    /**
     * 状态变更记录
     */
    @GetMapping("/statuslog")
    public ResponseEntity<Object> getStatusLog(@RequestParam("id") String id) {
        return new ResponseEntity<>(contractChangeStatusLogService.list(new LambdaQueryWrapper<ContractChangeStatusLog>()
                .eq(ContractChangeStatusLog::getContractId,id)
                .orderByAsc(ContractChangeStatusLog::getChangeTime)
        ),HttpStatus.OK);
    }

    /**
     * 获取自己公司
     */
//    @AnonymousAccess
    @GetMapping("/me/list")
    public ResponseEntity<Object> getMeCompanyList(@PageableDefault Pageable pageable) {
        return new ResponseEntity<>(new ArrayList<>(),HttpStatus.OK);
    }

    /**
     * 根据模板ID获取条款
     */
//    @AnonymousAccess
    @GetMapping("/tpl")
    public Object getTplById(@RequestParam("tplId") String tplId){
        return ctrtContractService.selectClausesByTplId(tplId);
    }

//    @AnonymousAccess
    @PostMapping("/save")
    public Object save(@RequestBody @Validated ContractDTO res) {
        ctrtContractService.saveEntity(res);
        return res;
    }

//    @AnonymousAccess
    @RequestMapping("/upload")
    public Object upload(@RequestParam("file") MultipartFile file) throws IOException {

        String fileName = IdUtil.getSnowflake(1, 1).nextIdStr();
        String ossKey = StrUtil.concat(true,"contract","/", fileName, ExtUtil.getExtname(file));
        ResultFile resultFile = ossObject.uploadFileByMultipartFile(ossKey, file, null);

        log.info("上传图片到云服务器！url = " + resultFile.getUrlHttps());
        return ResponseMapBuilder.newBuilder().putSuccess()
                .put("name",file.getOriginalFilename())
                .put("url", resultFile.getUrlHttps())
                .getResult();
    }

    private Map<Integer,List<TplContractDTO>> getContractTplByModule(List<Integer> modules) {
        LambdaQueryWrapper<TplContract> queryWrapper = new LambdaQueryWrapper<TplContract>();
        if (modules != null && modules.size() > 0) {
            queryWrapper.in(TplContract::getModule, modules);
        }
        return ctrtTplContractService.list(queryWrapper)
                .stream().map(item -> {
                    TplContractDTO dto = new TplContractDTO();
                    BeanUtils.copyProperties(item,dto);
                    dto.setContractTypeName(ContractTypeEnums.parseName(item.getContractType()));
                    dto.setContractTypeCode(Optional.ofNullable(ContractTypeEnums.parse(item.getContractType())).map(ContractTypeEnums::getCode).orElse(null));
                    return dto;
                }).collect(Collectors.groupingBy(TplContractDTO::getModule));
    }

    @GetMapping("/download")
    public void download(HttpServletResponse response, CtrtContractQueryCriteria criteria) throws IOException {
        ctrtContractService.download(ctrtContractService.queryAll(criteria),response);
    }

    // ===================================================================================
    // ===================================================================================
    // ===================================================================================

    @GetMapping
    @ApiOperation("查询合同")
    public ResponseEntity<Object> query(CtrtContractQueryCriteria criteria, Pageable pageable){
        return new ResponseEntity<>(ctrtContractService.queryAll(criteria,pageable),HttpStatus.OK);
    }

    @PostMapping
    @ApiOperation("新增合同")
    public ResponseEntity<Object> create(@Validated @RequestBody CtrtContract resources){
        return new ResponseEntity<>(ctrtContractService.save(resources),HttpStatus.CREATED);
    }

    @PutMapping
    @ApiOperation("修改合同")
    public ResponseEntity<Object> update(@Validated @RequestBody CtrtContract resources){
        ctrtContractService.updateById(resources);
        return new ResponseEntity<>(HttpStatus.NO_CONTENT);
    }

    @ApiOperation("删除合同")
    @DeleteMapping
    public ResponseEntity<Object> delete(@RequestBody String[] ids) {
        ctrtContractService.removeByIds(Arrays.asList(ids));
        return new ResponseEntity<>(HttpStatus.OK);
    }

    @GetMapping("/getContractByIds")
    public ResponseEntity<Object> getContractByIds(@RequestParam("ids") String ids) {
        Assert.notBlank(ids);
        List<CtrtContract> contracts = ctrtContractService.listByIds(StrUtil.split(ids, ',', -1));
        return new ResponseEntity<>(contracts,HttpStatus.OK);
    }

//    @GetMapping("/shipsContractData")
//    public ResponseEntity<Object> shipsContractData() {
//        return new ResponseEntity<>(contractService.shipsContractData(),HttpStatus.OK);
//    }

//    @PostMapping("/saveShipContract")
//    public ResponseEntity<Object> saveShipContract(@RequestBody @Validated ShipContract shipContract) {
//        contractService.saveShipContractData(shipContract);
//        return new ResponseEntity<>(HttpStatus.OK);
//    }

    /**
     * 合同作废
     */
    @PostMapping("/invalid")
    public ResponseEntity<Object> invalid(@RequestBody CtrtContract entity) {
        Assert.notBlank(entity.getId(),"合同ID不能为空");
        ctrtContractService.update(new LambdaUpdateWrapper<>(CtrtContract.class)
                .eq(CtrtContract::getId,entity.getId())
                .set(CtrtContract::getStatus, ContractStatusEnums.INVALID.getValue())
        );
        return new ResponseEntity<>(HttpStatus.OK);
    }

}