package com.zthzinfo.contract.handler.exchange;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.fastjson.JSONObject;
import com.itextpdf.text.*;
import com.itextpdf.text.pdf.BaseFont;
import com.itextpdf.text.pdf.PdfPCell;
import com.itextpdf.text.pdf.PdfPTable;
import com.zthzinfo.contract.domain.ContractClauses;
import com.zthzinfo.contract.enums.ShareTypeEnums;
import com.zthzinfo.contract.model.dto.ContractClausesFormDTO;
import com.zthzinfo.contract.model.dto.ContractDTO;
import com.zthzinfo.contract.service.CtrtContractService;
import org.apache.commons.lang3.StringEscapeUtils;
import org.springframework.expression.Expression;
import org.springframework.expression.common.TemplateParserContext;
import org.springframework.expression.spel.standard.SpelExpressionParser;

import javax.annotation.Resource;
import java.time.LocalDate;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

/**
 * 业务合同 exchange 抽象类
 *
 * @author zhaoey
 * @since 2021/12/20
 */
public abstract class AbstractBusinessContractExchange extends AbstractFileExchange<ContractDTO> {

    @Resource
    CtrtContractService ctrtContractService;

    @Override
    protected ContractDTO getData(String id) {
        return ctrtContractService.selectById(id);
    }

    @Override
    protected void assembleDocument(Document document, ContractDTO data, boolean isDiffStandard, String shareTypeKey) throws Exception {
        ContractClausesFormDTO form = new ContractClausesFormDTO();
        if (StrUtil.isNotBlank(data.getClausesForm())) {
            form = JSONObject.parseObject(StringEscapeUtils.unescapeJava(data.getClausesForm()), ContractClausesFormDTO.class);
        }

        //字体类型为宋体
        BaseFont Chinese = BaseFont.createFont("STSong-Light","UniGB-UCS2-H", BaseFont.NOT_EMBEDDED);
        Font font15 = new Font(Chinese, 15, Font.BOLD, BaseColor.BLACK);
        Font fontNormalBold = new Font(Chinese, 10, Font.BOLD, BaseColor.BLACK);
        Font fontNormal = new Font(Chinese, 10, Font.NORMAL, BaseColor.BLACK);
        Font fontNormalRed = new Font(Chinese, 10, Font.NORMAL, BaseColor.RED);
        BaseColor nonStandardColor = new BaseColor(255,99,71);
        Font fontNormalBoldNonStandard = new Font(Chinese, 10, Font.BOLD, nonStandardColor);
        Font fontNormalNonStandard = new Font(Chinese, 10, Font.NORMAL, nonStandardColor);

        // 空行
        Paragraph blank = new Paragraph(" ");

        // 标题
        addParagraph(document,data.getContractTitle(),font15, Element.ALIGN_CENTER);

        // 空行
//        document.add(blank);

        // 头部
        List<ContractClauses> clausesTops = data.getClausesTops();
        if (CollUtil.isNotEmpty(clausesTops)) {
            for (ContractClauses clausesTop : clausesTops) {
                boolean isSignDate=false;
                String tpl = clausesTop.getContent();
                String whiteStr = "consensusDate";
                String whiteSignStr = "signDate";
                if(StrUtil.contains(tpl,whiteStr)){
                    isSignDate=true;
                    // LocalDate format.
                    tpl = new StringBuffer(tpl).insert(tpl.indexOf(whiteStr)+whiteStr.length(),".toString()").toString();
                }
                if(StrUtil.contains(tpl,whiteSignStr) && data.getSignDate()!=null){
                    isSignDate=true;
                    // LocalDate format.
                    tpl = new StringBuffer(tpl).insert(tpl.indexOf(whiteSignStr)+whiteSignStr.length(),".toString().substring(0,10)").toString();
                }
                String content = parseContent(data, tpl);
                // 日期比较
                // data.getSignDate() 比当前日期小，显示红色
                if(isSignDate && shareTypeKey.equals(ShareTypeEnums.IMAGE.getKey()) && data.getConsensusDate()!=null && data.getConsensusDate().isBefore(LocalDate.now())){
                    addParagraph(document,clausesTop.getContentTitle(),content,fontNormalBold,fontNormalRed,Element.ALIGN_LEFT);
                }else{
                    addParagraph(document,clausesTop.getContentTitle(),content,fontNormalBold,fontNormal,Element.ALIGN_LEFT);
                }

            }
        }

        // 主要条款
        List<ContractClauses> clauses = data.getClauses();
        if (CollUtil.isNotEmpty(clauses)) {
            for (int i = 0; i < clauses.size(); i++) {
                ContractClauses item = clauses.get(i);
                String content = parseContent(form, item.getContent());
                // 非标合同标识颜色
                if(item.getStandard() || !isDiffStandard) {
                    // 标准条款
                    addParagraph(document,(i+1) + "."+ Optional.ofNullable(item.getContentTitle()).orElse(""),content,fontNormalBold,fontNormal,Element.ALIGN_LEFT);
                } else {
                    // 非标条款
                    addParagraph(document,(i+1) + "."+Optional.ofNullable(item.getContentTitle()).orElse(""),content,fontNormalBoldNonStandard,fontNormalNonStandard,Element.ALIGN_LEFT);
                }
                if (item.getShowTable() != null) {
//                    if (StrUtil.equals(item.getShowTable(),getShipName())) {
//                        shipTable(document,fontNormal,form);
//                    } else
                    if (StrUtil.equals(item.getShowTable(),"notice")) {
                        noticeTable(document,fontNormal,form);
                    } else {
                        shipTable(document,fontNormal,form);
                    }
                }
            }
        }

        // 底部
        List<ContractClauses> clausesBottoms = data.getClausesBottoms();
        if (CollUtil.isNotEmpty(clausesBottoms)) {
            for (ContractClauses item : clausesBottoms) {
                PdfPTable tableSignature = new PdfPTable(4);
                tableSignature.setWidthPercentage(100);
                cellItem(item.getContentTitle(),tableSignature,fontNormal,cell -> {
                    cell.setIndent(24);
                    cell.setBorderWidth(0);
                    cell.setHorizontalAlignment(Element.ALIGN_LEFT);
                    cell.setColspan(3);
                });
                cellItem(item.getContent(),tableSignature,fontNormal,cell -> {
                    cell.setBorderWidth(0);
                    cell.setHorizontalAlignment(Element.ALIGN_LEFT);
                });
                document.add(tableSignature);
            }
        }
    }

    protected void shipTable(Document document, Font font, ContractClausesFormDTO dto) throws DocumentException {
        basicShipTable(document,font,dto,table -> {
            cellItem("货名",table,font);
            cellItem("货重（吨）",table,font);
            cellItem("件数",table,font);
            cellItem("包装",table,font);
            cellItem("散杂体积(m³)",table,font);
            cellItem("运价（元/吨）",table,font);
            cellItem("开票类型",table,font);
            cellItem(dto.getGoodsName(),table,font);
            cellItem(dto.getGoodsWeight(),table,font);
            cellItem(dto.getQuantity(),table,font);
            cellItem(dto.getPack(),table,font);
            cellItem(dto.getBulkVolume(),table,font);
            cellItem(dto.getFreightRate(),table,font);
            cellItem(dto.getInvoiceType(),table,font);
            // 循环添加船价信息
            if (CollUtil.isNotEmpty(dto.getShipPriceList())) {
                for (ContractClausesFormDTO.ShipPrice shipPrice : dto.getShipPriceList()) {
                    cellItem(shipPrice.getGoodsName(), table, font);
                    cellItem(shipPrice.getGoodsWeight(), table, font);
                    cellItem(shipPrice.getQuantity(), table, font);
                    cellItem(shipPrice.getPack(), table, font);
                    cellItem(shipPrice.getBulkVolume(), table, font);
                    cellItem(shipPrice.getFreightRate(), table, font);
                    cellItem(shipPrice.getInvoiceType(), table, font);
                }
            }
        });
    }
    protected void noticeTable(Document document, Font font, ContractClausesFormDTO dto) throws DocumentException {
        // 创建表格对象
        PdfPTable table = new PdfPTable(8);
        table.setSpacingBefore(5);
        // 表格宽度
        table.setWidthPercentage(100);

        // 托运人
        cellItem("承租人确认联系方式与通讯地址",table,font,cell -> {
            cell.setColspan(4);
        });
        cellItem("出租人确认联系方式与通讯地址",table,font,cell -> {
            cell.setColspan(4);
        });
        cellItem("传真号",table,font);
        cellItem(dto.getLeaseeFaxno(),table,font);
        cellItem("邮箱",table,font);
        cellItem(dto.getLeaseeEmail(),table,font);
        cellItem("传真号",table,font);
        cellItem(dto.getLeaserFaxno(),table,font);
        cellItem("邮箱",table,font);
        cellItem(dto.getLeaserEmail(),table,font);
        cellItem("微信名",table,font);
        cellItem(dto.getLeaseeWechatName(),table,font);
        cellItem("微信号",table,font);
        cellItem(dto.getLeaseeWechatID(),table,font);
        cellItem("微信名",table,font);
        cellItem(dto.getLeaserWechatName(),table,font);
        cellItem("微信号",table,font);
        cellItem(dto.getLeaserWechatName(),table,font);
        cellItem("地址及收件人",table,font);
        cellItem(dto.getLeaseeContact(),table,font,cell -> cell.setColspan(3));
        cellItem("地址及收件人",table,font);
        cellItem(dto.getLeaserContact(),table,font,cell -> cell.setColspan(3));

        document.add(table);
    }

    protected void basicShipTable(Document document, Font font, ContractClausesFormDTO dto, Consumer<PdfPTable> consumer) throws DocumentException {
        basicShipTable(document,font,dto,table -> {
            cellItem("订金(元)",table,font);
            cellItem(dto.getDeposit(),table,font);
            cellItem("订金支付时间",table,font);
            cellItem(Optional.ofNullable(dto.getDepositPayTime()).map(LocalDate::toString).map(item -> item + "前").orElse(null),table,font);
            cellItem("滞期费率(元/吨天)",table,font);
            cellItem(dto.getDemurrageRate(),table,font,cell -> cell.setColspan(2));
        },consumer);
    }
    protected void basicShipTable(Document document, Font font, ContractClausesFormDTO dto, Consumer<PdfPTable> consumer1, Consumer<PdfPTable> consumer2) throws DocumentException {
        // 创建表格对象
        PdfPTable table = new PdfPTable(8);
        table.setSpacingBefore(5);
        table.setTotalWidth(550);
        table.setLockedWidth(true);
        float cellWidth = (float)(530/7);
        table.setWidths(new float[]{20f,cellWidth,cellWidth,cellWidth,cellWidth,cellWidth + 10,cellWidth,cellWidth-10});
        // 表格宽度
        table.setWidthPercentage(100);

        cellItem("船舶规范",table,font,cell -> {
            cell.setRowspan(4);
            cell.setPaddingLeft(5);
            cell.setPaddingRight(5);
        });
        cellItem("船名",table,font);
        cellItem(dto.getShipName(),table,font,cell -> cell.setColspan(2));
        cellItem("载重吨",table,font);
        cellItem(dto.getDeadWeight(),table,font);
        cellItem("长/高/宽(m)",table,font);
        cellItem(dto.getShipLengthHeightWidth(),table,font);
        cellItem("船籍港",table,font);
        cellItem(dto.getRegisterPort(),table,font,cell -> cell.setColspan(2));
        cellItem("总吨/净吨",table,font);
        cellItem(dto.getTon(),table,font);
        cellItem("满载吃水(m)",table,font);
        cellItem(dto.getHeavyDraft(),table,font);
        cellItem("建造时间",table,font);
        cellItem(dto.getBuildTime(),table,font,cell -> cell.setColspan(2));
        cellItem("总舱容(m³)",table,font);
        cellItem(dto.getHatchVolume(),table,font);
        cellItem("空载吃水(m)",table,font);
        cellItem(dto.getLightDraft(),table,font);
        cellItem("航 次",table,font);
        cellItem(dto.getVoyage(),table,font,cell -> cell.setColspan(2));
        cellItem("吊杆数/负荷",table,font);
        cellItem(dto.getDerrickNumber(),table,font);
        cellItem("舱口数",table,font);
        cellItem(dto.getHatchCount(),table,font);
        int rs;
        if(dto.getShipPriceList() != null && dto.getShipPriceList().size() > 0){
            rs = dto.getShipPriceList().size();
        } else {
            rs = 0;
        }
        cellItem("货运约定",table,font,cell -> {
            cell.setRowspan(5+rs);
            cell.setPaddingLeft(5);
            cell.setPaddingRight(5);
        });
        cellItem("起运港",table,font);
        cellItem(dto.getStartPort(),table,font,cell -> cell.setColspan(2));
        cellItem("目的港",table,font);
        cellItem(dto.getEndPort(),table,font,cell -> cell.setColspan(3));
        cellItem("受载日期",table,font);
        cellItem(dto.getPlanLoadingDate(),table,font,cell -> cell.setColspan(2));
        cellItem("抵运日期",table,font);
        cellItem(dto.getArrivalDate(),table,font,cell -> cell.setColspan(3));

        consumer1.accept(table);
        consumer2.accept(table);

        document.add(table);
    }

    public static <T> String parseContent(T dto,String tpl) {
        try {
            // 占位符如果为空的情况下，使用 / 进行填充
            tpl = tpl.replaceAll("#\\{\\s*(\\w+)\\s*}", "#{$1?:' / '}");
            TemplateParserContext context = new TemplateParserContext();
            SpelExpressionParser parser = new SpelExpressionParser();
            Expression expression = parser.parseExpression(tpl,context);
            return expression.getValue(dto,String.class);
        }catch (Exception e) {
            e.printStackTrace();
            return null;
        }
    }
    protected void addParagraph(Document document,String content,Font font,int alignment) throws DocumentException {
        addParagraph(document,null,content,null,font,alignment);
    }
    protected void addParagraph(Document document,String contentTitle,String content,Font titleFont,Font font,int alignment) throws DocumentException {
        Paragraph paragraph = new Paragraph(); //创建段落对象
        if(StrUtil.isNotBlank(contentTitle)) {
            Chunk chunkTitle = new Chunk(contentTitle,titleFont);
            paragraph.add(chunkTitle);
        }
        if (StrUtil.isNotBlank(content)) {
            Chunk chunkContent = new Chunk(content,font);
            paragraph.add(chunkContent);
        }
        paragraph.setAlignment(alignment);
        document.add(paragraph);
    }
    protected void cellItem(Object value, PdfPTable table, Font font12) {
        cellItem(value,table,font12,cell->{});
    }
    protected void cellItem(Object value, PdfPTable table, Font font12, Consumer<PdfPCell> consumer) {
        PdfPCell cell = new PdfPCell();
        cell.setMinimumHeight(20);
        cell.setPhrase(new Paragraph(Optional.ofNullable(value).map(String::valueOf).orElse("/"), font12));
        cell.setHorizontalAlignment(Element.ALIGN_CENTER);
        cell.setVerticalAlignment(Element.ALIGN_MIDDLE);
        consumer.accept(cell);
        table.addCell(cell);
    }

}
