package com.zthzinfo.contract.handler.exchange;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.StrUtil;
import com.itextpdf.text.Document;
import com.itextpdf.text.PageSize;
import com.itextpdf.text.pdf.PdfWriter;
import com.zthzinfo.aliyun.oss.OSSObject;
import com.zthzinfo.aliyun.oss.ResultFile;
import com.zthzinfo.contract.enums.ShareTypeEnums;
import com.zthzinfo.contract.handler.IContractFileExchange;
import com.zthzinfo.contract.model.dto.FileParams;
import com.zthzinfo.contract.utils.MockMultipartFile;
import lombok.extern.slf4j.Slf4j;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.rendering.PDFRenderer;
import org.springframework.core.env.Environment;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.StringJoiner;

/**
 * @author zhaoeryu
 * @since 2021/9/18
 */
@Slf4j
public abstract class AbstractFileExchange<T> implements IContractFileExchange {

    @Resource
    Environment environment;
    @Resource
    OSSObject ossObject;

    @Override
    public String exchange(FileParams fileParam) {
        // 获取业务数据
        T data = fileParam.dataToT();
        if (!fileParam.isUseData()) {
            data = getData(fileParam.getId());
        }

        // 创建临时文件
        String fileName = genFileName();
        String tmpFilePath = getTmpFilePath(fileParam.getModule());
        FileUtil.mkdir(tmpFilePath);
        String tmpPdfFilePath = StrUtil.concat(true,tmpFilePath, File.separator,fileName, ".pdf");
        log.debug("临时文件保存路径：{}",tmpPdfFilePath);
        File tmpPdfFile = FileUtil.touch(tmpPdfFilePath);

        try (FileOutputStream fos = new FileOutputStream(tmpPdfFile)){
            // 创建pdf文档对象
            Document document = new Document(PageSize.A4,20,20,20,20);

            // 将文件输出流与pdf对象，进行关联
            PdfWriter.getInstance(document, fos);

            // 打开文档
            document.open();

            // 拼装文档内容
            assembleDocument(document,data,fileParam.isDiffStandard(),fileParam.getType());

            // 关闭文档
            document.close();

            // 上传文件
            return upload(tmpPdfFile, tmpFilePath, fileParam);
        } catch (Exception e) {
            e.printStackTrace();
            log.error("生成PDF并上传到OSS异常,{}",e.getMessage());
            throw new RuntimeException(e.getMessage());
        } finally {
            try {
                // 删除临时文件
                FileUtil.del(tmpPdfFile);
            } catch (Exception e) {
                e.printStackTrace();
                log.error("删除临时文件异常,{}",e.getMessage());
            }
        }
    }

    /**
     * 获取临时文件目录
     */
    protected String getTmpFilePath(String module) {
        return new StringJoiner(File.separator)
                .add(System.getProperty("java.io.tmpdir"))
                .add(StrUtil.blankToDefault(environment.getProperty("spring.application.name"), "base-service"))
                .add(module).toString();
    }

    /**
     * 上传文件
     */
    protected String upload(File tmpPdfFile, String tmpFilePath, FileParams fileParam) throws IOException {
        if (ShareTypeEnums.PDF.getKey().equals(fileParam.getType())) {
            return uploadPDF(tmpPdfFile, fileParam.getModule());
        }
        if(ShareTypeEnums.IMAGE.getKey().equals(fileParam.getType())) {
            return uploadImage(tmpPdfFile, tmpFilePath, fileParam);
        }
        throw new IllegalArgumentException("错误的分享类型："+fileParam.getType());
    }

    /**
     * 上传PDF
     */
    private String uploadPDF(File tmpPdfFile, String module) throws IOException {
        try (FileInputStream fileInputStream = new FileInputStream(tmpPdfFile)){
            return uploadFileToOSS(fileInputStream, ".pdf","application/pdf",module);
        }
    }

    /**
     * 上传图片
     */
    private String uploadImage(File tmpPdfFile, String tmpFilePath, FileParams fileParam) throws IOException {
        // PDF转图片
        try(PDDocument tmpDoc = PDDocument.load(tmpPdfFile);) {
            PDFRenderer renderer = new PDFRenderer(tmpDoc);

            int pages = tmpDoc.getNumberOfPages();
            List<BufferedImage> images = new ArrayList<>();
            if (pages < 2) {
                // 单图
                images.add(getSingleImageFile(renderer));
            } else {
                if (fileParam.isGenLongImageFile()) {
                    // 长图模式
                    images.add(getLongImageFile(renderer,pages));
                } else {
                    // 多图模式
                    images = getMultiImages(renderer,pages);
                }
            }

            StringJoiner resultSj = new StringJoiner("|");
            for (BufferedImage image : images) {
                String tmpImageFilePath = StrUtil.concat(true,tmpFilePath, File.separator,genFileName(), ".png");
                File tmpImageFile = new File(tmpImageFilePath);
                ImageIO.write(image, "PNG", tmpImageFile);
                try (FileInputStream imgFIS = new FileInputStream(tmpImageFile)){
                    // 上传到阿里云
                    String imgUrl = uploadFileToOSS(imgFIS, ".png", "image/png", fileParam.getModule());
                    resultSj.add(imgUrl);
                } finally {
                    try {
                        // 删除临时文件
                        FileUtil.del(tmpImageFile);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                }
            }
            return resultSj.toString();
        }
    }

    /** 生成文件名 */
    private String genFileName() {
        return IdUtil.getSnowflake(1, 1).nextIdStr();
    }

    /** 单张图片 */
    private BufferedImage getSingleImageFile(PDFRenderer renderer) throws IOException {
        return renderer.renderImage(0, 2f);
    }
    /** 长图片 */
    private BufferedImage getLongImageFile(PDFRenderer renderer, int pages) throws IOException {
        // 定义宽度
        int width = 0;
        // 保存一张图片中的RGB数据
        int[] singleImgRGB;
        // 定义高度，后面用于叠加
        int shiftHeight = 0;
        //保存每张图片的像素值
        BufferedImage imageResult = null;
        for (int i = 0; i < pages; i++) {
            BufferedImage image = renderer.renderImage(i, 2f);
            int imageHeight = image.getHeight();
            int imageWidth = image.getWidth();
            if (i == 0) {
                //计算高度和偏移量
                //使用第一张图片宽度;
                width = imageWidth;
                // 保存每页图片的像素值
                imageResult = new BufferedImage(width, imageHeight * pages, BufferedImage.TYPE_INT_RGB);
            } else {
                // 将高度不断累加
                shiftHeight += imageHeight;
            }
            singleImgRGB = image.getRGB(0, 0, width, imageHeight, null, 0, width);
            imageResult.setRGB(0, shiftHeight, width, imageHeight, singleImgRGB, 0, width);
        }
        return imageResult;
    }
    /**
     * 多图
     */
    private List<BufferedImage> getMultiImages(PDFRenderer renderer, int pages) throws IOException {
        List<BufferedImage> images = new ArrayList<>();
        for (int i = 0; i < pages; i++) {
            images.add(renderer.renderImage(i, 1f));
        }
        return images;
    }

    /**
     * 上传文件到OSS
     */
    private String uploadFileToOSS(FileInputStream fileInputStream, String ext, String contentType, String module) throws IOException {
        String fileName = genFileName();
        String ossKey = StrUtil.concat(true,module,"/", fileName, ext);
        MultipartFile multipartFile = new MockMultipartFile(fileName,fileName,contentType, fileInputStream);
        ResultFile resultFile = ossObject.uploadFileByMultipartFile(ossKey, multipartFile, null);
        log.info("上传文件到OSS,url={} ", resultFile.getUrl());
        return resultFile.getUrlHttps();
    }

    /**
     * 获取模板表单的填充数据
     */
    protected abstract T getData(String id);

    /**
     * 组装文档对象
     */
    protected abstract void assembleDocument(Document document, T data,boolean isDiffStandard,String shareTypeKey) throws Exception;
}
