package com.zthzinfo.contract.service.impl;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.StrUtil;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.metadata.style.WriteCellStyle;
import com.alibaba.excel.write.metadata.style.WriteFont;
import com.alibaba.excel.write.style.HorizontalCellStyleStrategy;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.github.pagehelper.PageInfo;
import com.google.common.collect.Lists;
import com.zthzinfo.contract.base.BaseServiceImpl;
import com.zthzinfo.contract.base.QueryHelpPlus;
import com.zthzinfo.contract.domain.CtrtContract;
import com.zthzinfo.contract.domain.ContractClauses;
import com.zthzinfo.contract.domain.TplContract;
import com.zthzinfo.contract.domain.TplContractClauses;
import com.zthzinfo.contract.enums.*;
import com.zthzinfo.contract.handler.dispatcher.ContractFileDispatcher;
import com.zthzinfo.contract.handler.dispatcher.ContractProcessGetChartererDispatcher;
import com.zthzinfo.contract.handler.dispatcher.ContractProcessStatusDispatcher;
import com.zthzinfo.contract.mapper.CtrtContractMapper;
import com.zthzinfo.contract.model.dto.*;
import com.zthzinfo.contract.service.*;
import com.zthzinfo.contract.service.dto.ContractDto;
import com.zthzinfo.contract.service.dto.CtrtContractQueryCriteria;
import com.zthzinfo.contract.service.mapstruct.ContractSMapper;
import com.zthzinfo.contract.utils.FileUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.usermodel.*;
import org.springframework.beans.BeanUtils;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

/**
* @description 服务实现
* @author zhaoey
* @date 2021-12-14
**/
@Slf4j
@Service
public class CtrtContractServiceImpl extends BaseServiceImpl<CtrtContractMapper, CtrtContract> implements CtrtContractService {

    @Resource
    private ContractSMapper contractSMapper;
    @Resource
    private CtrtContractMapper ctrtContractMapper;
    @Resource
    CtrtContractClausesService clausesService;
    @Resource
    private CtrtTplContractClausesService ctrtTplContractClausesService;
    @Resource
    private CtrtTplContractService ctrtTplContractService;
    @Resource
    CtrtIdGenService ctrtIdGenService;
    @Resource
    ContractFileDispatcher contractFileDispatcher;
//    @Resource
//    WxDepartmentService wxDepartmentService;
//    @Resource
//    SysExternalCustomerService sysExternalCustomerService;


    @Resource
    ContractProcessStatusDispatcher contractProcessStatusDispatcher;
    @Resource
    ContractProcessGetChartererDispatcher contractProcessGetChartererDispatcher;

    @Override
    public Map<String,Object> queryAll(CtrtContractQueryCriteria criteria, Pageable pageable){
        getPage(pageable);
        PageInfo<ContractDto> page = new PageInfo<>(queryAll(criteria));
        return toMap(page);
    }

    @Override
    public List<ContractDto> queryAll(CtrtContractQueryCriteria criteria){
        return listToDto(list(QueryHelpPlus.getPredicate(CtrtContract.class, criteria)),contractSMapper);
    }

    @Override
    public ContractDTO selectById(String id) {
        CtrtContract entity = getById(id);
        Assert.notNull(entity,"未找到该合同");
        ContractDTO dto = selectClauses(entity.getTplId());
        BeanUtils.copyProperties(entity,dto);
        dto.setContractTypeName(ContractTypeEnums.parseName(entity.getContractType()));
        dto.setSpStatusName(ProcessStatusEnum.getDescByKey(entity.getSpStatus()));
        return dto;
    }

    private ContractDTO selectClauses(String tplId) {
        List<ContractClauses> clauses = ctrtTplContractClausesService.list(new LambdaQueryWrapper<TplContractClauses>().eq(TplContractClauses::getTplId,tplId)
                        .orderByAsc(TplContractClauses::getSort))
                .stream().map(item -> {
                    ContractClauses clause = new ContractClauses();
                    clause.setContent(item.getContent());
                    clause.setStandard(item.getStandard());
                    clause.setTplClauseId(item.getId());
                    clause.setEnableChange(item.getEnableChange());
                    clause.setShowTable(item.getShowTable());
                    clause.setContentTitle(item.getContentTitle());
                    clause.setSort(item.getSort());
                    clause.setClausesType(item.getClausesType());
                    return clause;
                }).collect(Collectors.toList());

        Map<Integer, List<ContractClauses>> table = clauses.stream().collect(Collectors.groupingBy(ContractClauses::getClausesType));

        ContractDTO dto = new ContractDTO();
        dto.setClauses(Optional.ofNullable(table).map(item -> item.get(ContractClausesTypeEnums.NORMAL.getKey())).orElse(new ArrayList<>()));
        dto.setClausesTops(Optional.ofNullable(table).map(item -> item.get(ContractClausesTypeEnums.TOP.getKey())).orElse(new ArrayList<>()));
        dto.setClausesBottoms(Optional.ofNullable(table).map(item -> item.get(ContractClausesTypeEnums.BOTTOM.getKey())).orElse(new ArrayList<>()));
        return dto;
    }

    @Override
    public ContractDTO selectClausesByTplId(String tplId) {
        ContractDTO dto = selectClauses(tplId);
        TplContract tpl = ctrtTplContractService.getById(tplId);
        dto.setTplSchema(tpl.getFormSchema());
        dto.setContractType(tpl.getContractType());
        return dto;
    }

    @Transactional(rollbackFor = Exception.class)
    @Override
    public void saveEntity(ContractDTO res) {
        // 生成合同号
        incrGenContractNo(res);

        // 上传文件
        contractFileUpload(res);

        if (StrUtil.isNotBlank(res.getId())) {
            // 如果有合同ID，需要把之前的条款删除重新保存
            // 把之前的条款删除
            clausesService.remove(new LambdaQueryWrapper<ContractClauses>().eq(ContractClauses::getContractId, res.getId()));
        }

        // 合同基本信息
        saveOrUpdate(res);

        // 合同条款
        List<ContractClauses> list = CollUtil.newArrayList();
        if (CollUtil.isNotEmpty(res.getClausesTops())) {
            list.addAll(res.getClausesTops());
        }
        if (CollUtil.isNotEmpty(res.getClauses())) {
            list.addAll(res.getClauses());
        }
        if (CollUtil.isNotEmpty(res.getClausesBottoms())) {
            list.addAll(res.getClausesBottoms());
        }
        if (CollUtil.isEmpty(list)) {
            return;
        }

        list = list.stream().peek(item -> {
            item.setId(null);
            item.setContractId(res.getId());
        }).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(list)) {
            clausesService.saveBatch(list);
        }
    }

    /**
     * 自增生成合同号
     * @param res
     */
    private void incrGenContractNo(ContractDTO res) {
        if (res.getSignDate() == null || StrUtil.isNotBlank(res.getContractNo())) {
            log.info("签订日期为空或者已有合同号不设置合同号");
            return;
        }
//        if (StrUtil.isBlank(res.getCompanyCode())) {
//            String companyId = BooleanUtil.toString(res.getMyIsPartyA(), res.getPartyAId(), res.getPartyBId());
            // TODO 重新抽象接口
//            String code = Optional.ofNullable(sysExternalCustomerService.getById(companyId)).map(SysExternalCustomer::getSpare5).orElse(AppConstants.DEFAULT_COMPANY_CODE);
//            res.setCompanyCode(code);
//        }
        // TODO 重新抽象接口
//        String deptCode = Optional.ofNullable(wxDepartmentService.getCurrentUserDeptCode()).orElse(AppConstants.DEFAULT_DEPT_CODE);
        String contractNo = ctrtIdGenService.incrementAndGet(res.getContractNoFirstPrefix(), res.getContractNoSecondPrefix(), ContractTypeEnums.parse(res.getContractType()).getCode());
        res.setContractNo(contractNo);
    }

    private void contractFileUpload(ContractDTO res) {
        if (!res.getUseTpl()) {
            // 如果没使用模板，不进行处理
            return;
        }

        // 生成文件：非标条款不进行特殊处理
        FileParams fileParams = new FileParams();
        fileParams.setType(ShareTypeEnums.PDF.getKey());
        fileParams.setModule("contract");
        fileParams.setGenLongImageFile(true);
        fileParams.setUseData(true);
        fileParams.setData(res);
        fileParams.setDiffStandard(false);
        fileParams.setContractType(res.getContractType());
        String url = contractFileDispatcher.dispatch(fileParams);
        res.setFileUrl(url);
        res.setFileName(FileUtil.getName(url));

        // 预览图片：非标条款进行特殊处理
        fileParams.setType(ShareTypeEnums.IMAGE.getKey());
        fileParams.setDiffStandard(true);
        url = contractFileDispatcher.dispatch(fileParams);
        res.setPreviewFileUrl(url);
    }

    @Override
    public void download(List<ContractDto> dtos, HttpServletResponse response) throws IOException {
        List<ContractVo> voArrayList = new ArrayList<>();
        for (int i = 0; i < dtos.size(); i++) {
            ContractDto item = dtos.get(i);
            ContractVo vo = new ContractVo();
            vo.setNo((i+1)+"");
            vo.setContractNo(item.getContractNo());
            vo.setSignDate(Optional.ofNullable(item.getSignDate()).map(sd -> sd.format(DateTimeFormatter.ofPattern("yyyy年MM月dd日"))).orElse(null));
            vo.setContractType(item.getContractTitle());
            vo.setPartyAName(BooleanUtil.toString(Optional.ofNullable(item.getMyIsPartyA()).orElse(true),item.getPartyAName(),item.getPartyBName()));
            vo.setPartyBName(BooleanUtil.toString(Optional.ofNullable(item.getMyIsPartyA()).orElse(true),item.getPartyBName(),item.getPartyAName()));
            voArrayList.add(vo);
        }
        voArrayList.add(new ContractVo("","","","","","","","","","","","","","","","","审核（公司/部门负责人）：","",""));
        voArrayList.add(new ContractVo("备注","","","","","","","","","","","","","","","","","",""));

        // 头的策略
        WriteCellStyle headWriteCellStyle = new WriteCellStyle();
        // 背景设置为白色
        headWriteCellStyle.setFillPatternType(FillPatternType.SOLID_FOREGROUND);
        headWriteCellStyle.setFillForegroundColor(IndexedColors.WHITE.getIndex());
        WriteFont writeFont = new WriteFont();
        writeFont.setBold(false);
        writeFont.setFontHeightInPoints((short)11);
        headWriteCellStyle.setWriteFont(writeFont);

        // 内容的策略
        WriteCellStyle contentWriteCellStyle = new WriteCellStyle();
        //设置 垂直居中
        contentWriteCellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        //设置 水平居中
        contentWriteCellStyle.setHorizontalAlignment(HorizontalAlignment.CENTER);
        contentWriteCellStyle.setBorderBottom(BorderStyle.THIN);
        contentWriteCellStyle.setBorderTop(BorderStyle.THIN);
        contentWriteCellStyle.setBorderLeft(BorderStyle.THIN);
        contentWriteCellStyle.setBorderRight(BorderStyle.THIN);

        HorizontalCellStyleStrategy horizontalCellStyleStrategy = new HorizontalCellStyleStrategy(headWriteCellStyle,contentWriteCellStyle);

        try {
            EasyExcel.write(response.getOutputStream(), ContractVo.class)
                    .registerWriteHandler(horizontalCellStyleStrategy)
                    .head(head())
                    .sheet("统计报表").doWrite(voArrayList);
        } catch (Exception e) {

            e.printStackTrace();
        }
    }
    // 动态表头
    private List<List<String>> head() {
        List<List<String>> headTitles = CollUtil.newArrayList();
        String title = "公司合同台账管理表",empty="",empty1=" ";

        headTitles.add(Lists.newArrayList(title,"报送部门/公司：","填表人：","填表日期：","序号","序号"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"合同编号","合同编号"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"签订日期","签订日期"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"类型","类型"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"合同当事人","我方单位"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"合同当事人","他方单位"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"合同生效条件及日期","合同生效条件及日期"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"我方义务概述","我方义务概述"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"标的","标的"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"收付方式","收付方式"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"订金（预付款）","订金（预付款）"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"合同总价款","合同总价款"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"合同履行情况","合同履行情况"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"是否存在违约情形","是否存在违约情形"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"违约情形描述","违约情形描述"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"违约责任（违约金）","违约责任（违约金）"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"合同份数","合同份数"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"签约承办部门、经办人、联系方式","签约承办部门、经办人、联系方式"));
        headTitles.add(Lists.newArrayList(title,empty,empty1,empty,"备注","备注"));
        return headTitles;
    }

    /**
     * 更新签订日期
     */
    @Transactional(rollbackFor = Exception.class)
    @Override
    public String updateSignDate(ContractDTO res) {
        // 生成合同号
        incrGenContractNo(res);

        // 更新签订日期
        updateById(res);

        return res.getContractNo();
    }

    public PageInfo<ContractDto> searchContract(CtrtContractQueryCriteria criteria, Pageable pageable) {
        getPage(pageable);
        PageInfo<ContractDto> page = new PageInfo<>(queryAll(criteria));
        return page;
    }
}