package com.zthzinfo.sdks.netease.im.service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.zthzinfo.sdks.netease.common.utils.NEResquester;
import com.zthzinfo.sdks.netease.common.utils.NameValuePairBuilder;
import com.zthzinfo.sdks.netease.im.vo.NEIMUser;
import org.apache.http.NameValuePair;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * IM用户相关service
 * Created by sunzsh on 2016/12/15.
 */
public class NEIMUserService {
    private static final String USER_BASE_URL = NEResquester.BASE_URL + "/nimserver/user";
    private NEResquester requester;

    /**
     * 创建用户
     *
     * @param accid      云信ID，最大长度32字符，必须保证一APP内唯一（只允许字母、数字、半角下划线_、@、半角点以及半角-组成，不区分大小写，会统一小写处理，请注意以此接口返回结果中的accid为准）
     * @param token      云信ID可以指定登录token值，最大长度128字符，并更新，如果未指定，会自动生成token，并在创建成功后返回
     * @param name       云信ID昵称，最大长度64字符，用来PUSH推送时显示的昵称
     * @param jsonProps  json属性，第三方可选填，最大长度1024字符
     * @param headerIcon 云信ID头像URL，第三方可选填，最大长度1024
     * @return
     */
    public NEIMUser createUser(String accid, String token, String name, String jsonProps, String headerIcon) {
        if (accid == null) {
            throw new IllegalArgumentException("accid不能为空");
        }
        if (jsonProps != null) {
            if (jsonProps.length() > 1024) {
                throw new IllegalArgumentException("jsonProps长度不能超过1024");
            }
            try {
                JSON.parseObject(jsonProps);
            } catch (Exception e) {
                throw new IllegalArgumentException("jsonProps不是有效的JsonObject格式");
            }
        }

        // build参数集合
        List<NameValuePair> nvps = NameValuePairBuilder.newBuilder()
                .addParam("accid", accid)
                .addParam("token", token)
                .addParam("name", name)
                .addParam("props", jsonProps)
                .addParam("icon", headerIcon).getParams();

        JSONObject jsonObject = requester.getJSONObjectOnly200(USER_BASE_URL + "/create.action", nvps);
        if (jsonObject == null) {
            return null;
        }

        JSONObject info = jsonObject.getJSONObject("info");
        NEIMUser user = new NEIMUser();
        user.setAccid(info.getString("accid"));
        user.setToken(info.getString("token"));
        user.setName(info.getString("name"));
        return user;
    }

    /**
     * 获取用户信息
     * @param accid
     * @return
     */
    public NEIMUser getUserInfo(String accid) {
        if (accid == null) {
            return null;
        }
        List<NEIMUser> list = getUserInfos(new String[]{accid});
        if (list == null || list.size() == 0) {
            return null;
        }
        return list.get(0);
    }

    /**
     * 批量获取用户信息
     * @param accids 用户accid数组
     */
    public List<NEIMUser> getUserInfos(String[] accids) {
        if (accids == null) {
            return null;
        }
        if (accids.length == 0) {
            return null;
        }


        JSONArray ja = new JSONArray();
        for (String accid : accids) {
            ja.add(accid);
        }
        // build参数集合
        List<NameValuePair> nvps = NameValuePairBuilder.newBuilder()
                .addParam("accids", ja.toJSONString())
                .getParams();

        JSONObject jo = requester.getJSONObjectOnly200(USER_BASE_URL + "/getUinfos.action", nvps);
		if (jo == null) {
			return null;
		}
		List<NEIMUser> result = new ArrayList<>();
		JSONArray uinfos = jo.getJSONArray("uinfos");
        if (uinfos == null) {
            return result;
        }

        for (int i = 0; i < uinfos.size(); i++) {
            JSONObject userJson = uinfos.getJSONObject(i);
            result.add(new NEIMUser(userJson));
        }
        return result;
    }

	/**
	 * 更新用户信息
	 * @param accid 云信ID，最大长度32字符，必须保证一个APP内唯一
	 * @param jsonProps json属性，第三方可选填，最大长度1024字符
	 * @param token 云信ID可以指定登录token值，最大长度128字符
	 * @return 更新是否成功
	 */
	public boolean updateUserInfo(String accid, String jsonProps, String token){
		if (accid == null) {
			throw new IllegalArgumentException("accid不能为空");
		}
		if (jsonProps != null) {
			if (jsonProps.length() > 1024) {
				throw new IllegalArgumentException("jsonProps长度不能超过1024");
			}
			try {
				JSON.parseObject(jsonProps);
			} catch (Exception e) {
				throw new IllegalArgumentException("jsonProps不是有效的JsonObject格式");
			}
		}
		// build参数集合
		List<NameValuePair> nvps = NameValuePairBuilder.newBuilder()
				.addParam("accid", accid)
				.addParam("token", token)
				.addParam("props", jsonProps)
				.getParams();

		JSONObject jsonObject = requester.getJSONObjectOnly200(USER_BASE_URL + "/update.action", nvps);
		if (jsonObject == null) {
			return false;
		}
		return true;
	}

    public NEResquester getRequester() {
        return requester;
    }

    public void setRequester(NEResquester requester) {
        this.requester = requester;
    }
}
