package com.cxqm.xiaoerke.common.queue;

import com.cxqm.xiaoerke.common.utils.SpringContextHolder;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.context.ApplicationContext;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.*;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Created by sunzsh on 2017/3/15.
 */
public class RedisDelayQueue implements InitializingBean,DisposableBean {
	private RedisTemplate redisTemplate;
	private String key;
	private RedisConnectionFactory factory;
	private RedisConnection connection;//for blocking
	private BoundZSetOperations<String, RedisDelayQueueEntity> setOperations;//noblocking

	private Lock lock = new ReentrantLock();//基于底层IO阻塞考虑

	private RedisDelayQueueEntity delayQueueEntity; // 延迟消息载体
	private Thread listenerThread;

	private boolean isClosed;

	public void setRedisTemplate(RedisTemplate redisTemplate) {
		this.redisTemplate = redisTemplate;
	}

	public RedisDelayQueueEntity getDelayQueueEntity() {
		return delayQueueEntity;
	}

	public void setKey(String key) {
		this.key = key;
	}


	@Override
	public void afterPropertiesSet() throws Exception {
		factory = redisTemplate.getConnectionFactory();
		connection = RedisConnectionUtils.getConnection(factory);
		setOperations = redisTemplate.boundZSetOps(key);
		listenerThread = new ListenerThread();
		listenerThread.setDaemon(true);
		listenerThread.start();
	}



	/**
	 * 根据一个到期时间添加一个对象
	 */
	public void addByExpiredTime(RedisDelayQueueTiming queueTiming, Date expiredTime){
		setOperations.add(new RedisDelayQueueEntity(queueTiming.getQueueName(), queueTiming.getValue()), new Double(expiredTime.getTime()));
	}
	/**
	 * 根据一个到期时间添加一个对象
	 */
	public void addByAfterSecond(RedisDelayQueueTiming queueTiming, long second){
		addByExpiredTime(queueTiming, new Date(new Date().getTime() + second * 1000));
	}

	public void cancel(RedisDelayQueueTiming queueTiming) {
		setOperations.remove(new RedisDelayQueueEntity(queueTiming.getQueueName(), queueTiming.getValue()));

	}

	/**
	 * blocking
	 * remove and get first item from queue:BLPOP
	 * @return
	 */
	public List<RedisDelayQueueEntity> getListForTimeout() throws InterruptedException{
		lock.lockInterruptibly();
		try{
			Long currentTime = new Date().getTime();
			connection.multi();
			Set<RedisDelayQueueEntity> ts = setOperations.rangeByScore(0.d, new Double(currentTime));
			setOperations.removeRangeByScore(0.d, new Double(currentTime));
			connection.exec();
			if(CollectionUtils.isEmpty(ts)){
				return null;
			}
			List<RedisDelayQueueEntity> result = new ArrayList<RedisDelayQueueEntity>();
			for (RedisDelayQueueEntity t : ts) {
				result.add(t);
			}
			return result;
		} catch (Exception e) {
			connection.discard();
			return null;
		} finally {
			lock.unlock();
		}
	}


	@Override
	public void destroy() throws Exception {
		System.out.println("----destroy.... ");
		isClosed = true;
		shutdown();
		connection.getSentinelConnection().close();
		connection.shutdown();
		connection.close();
		RedisConnectionUtils.unbindConnection(factory);
		RedisConnectionUtils.releaseConnection(connection, factory);
	}

	private void shutdown(){
		try{
			listenerThread.interrupt();
		}catch(Exception e){
			//
			e.printStackTrace();
		}
	}

	class ListenerThread extends Thread {

		@Override
		public void run(){
			try{
				while(!isClosed){
					List<RedisDelayQueueEntity> ts = getListForTimeout();
					//逐个执行
					if(ts != null && ts.size() > 0){
						for (RedisDelayQueueEntity t : ts) {
							ApplicationContext applicationContext = SpringContextHolder.getApplicationContext();

							Object bean = applicationContext.getBean(t.getKey());
							if (bean == null || !(bean instanceof RedisQueue)) {
								continue;
							}
							RedisQueue redisQueue = ((RedisQueue) bean);
							redisQueue.pushFromHead(t.getValue());
						}

					}
					Thread.sleep(1000l);	// 由于getListForTimeout()并非阻塞模式，所以考虑性能，每次循环之后睡眠1s
				}
			}catch(InterruptedException e){
				//
			}
		}


	}

}
