package com.cxqm.xiaoerke.common.queue;

import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisConnectionFactory;
import org.springframework.data.redis.core.BoundListOperations;
import org.springframework.data.redis.core.RedisConnectionUtils;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * Created by xuebing on 2017/3/3.
 */
public class RedisQueue<T> implements InitializingBean,DisposableBean {
	private RedisTemplate redisTemplate;
	private String key;
	private int cap = Short.MAX_VALUE;//最大阻塞的容量，超过容量将会导致清空旧数据
	private byte[] rawKey;
	private RedisConnectionFactory factory;
	private RedisConnection connection;//for blocking
	private BoundListOperations<String, RedisQueueItem<T>> listOperations;//noblocking

	private Lock lock = new ReentrantLock();//基于底层IO阻塞考虑

	private RedisQueueListener<T> listener;//异步回调
	private Thread listenerThread;

	private boolean isClosed = false;

	public void setRedisTemplate(RedisTemplate redisTemplate) {
		this.redisTemplate = redisTemplate;
	}

	public void setListener(RedisQueueListener listener) {
		this.listener = listener;
	}
	//设置key的过期时间
	public void setKey(String key) {
		this.key = key;
	}

	//设置key过期时间


	@Override
	public void afterPropertiesSet() throws Exception {
		factory = redisTemplate.getConnectionFactory();
		connection = RedisConnectionUtils.getConnection(factory);
		rawKey = redisTemplate.getKeySerializer().serialize(key);
		listOperations = redisTemplate.boundListOps(key);
		if(listener != null){
			listenerThread = new ListenerThread();
			listenerThread.setDaemon(true);
			listenerThread.start();
		}
	}


	/**
	 * blocking
	 * remove and get last item from queue:BRPOP
	 * @return
	 */
	public RedisQueueItem<T> takeFromTail(int timeout) throws InterruptedException{
		lock.lockInterruptibly();
		try{
			List<byte[]> results = connection.bRPop(timeout, rawKey);
			if(CollectionUtils.isEmpty(results)){
				return null;
			}
			return (RedisQueueItem<T>)redisTemplate.getValueSerializer().deserialize(results.get(1));
		}finally{
			lock.unlock();
		}
	}

	public RedisQueueItem<T> takeFromTail() throws InterruptedException{
		return takeFromTail(0);
	}

	/**
	 * 从队列的头，插入
	 */
	public void pushFromHead(T value){
		listOperations.leftPush(new RedisQueueItem(value));
	}

	public void pushFromTail(T value){
		listOperations.rightPush(new RedisQueueItem(value));
	}

	public void pushShutdownSign() {
		listOperations.rightPush(RedisQueueItem.getShutdownSign());
	}

	/**
	 * noblocking
	 * @return null if no item in queue
	 */
	public RedisQueueItem<T> removeFromHead(){
		return listOperations.leftPop();
	}

	public RedisQueueItem<T> removeFromTail(){
		return listOperations.rightPop();
	}

	/**
	 * blocking
	 * remove and get first item from queue:BLPOP
	 * @return
	 */
	public RedisQueueItem<T> takeFromHead(int timeout) throws InterruptedException{
		lock.lockInterruptibly();
		try{
			List<byte[]> results = connection.bLPop(timeout, rawKey);

			if(CollectionUtils.isEmpty(results)){
				return null;
			}
			return (RedisQueueItem<T>)redisTemplate.getValueSerializer().deserialize(results.get(1));
		}finally{
			lock.unlock();
		}
	}

	public RedisQueueItem<T> takeFromHead() throws InterruptedException{
		return takeFromHead(0);
	}

	@Override
	public void destroy() throws Exception {
		isClosed = true;
		shutdown();
		connection.getSentinelConnection().close();
		connection.shutdown();
		connection.close();
		RedisConnectionUtils.unbindConnection(factory);
		RedisConnectionUtils.releaseConnection(connection, factory);
	}

	private void shutdown(){
		try{
			while(listenerThread.isAlive()) {
				System.out.println("push shutdown sign");
				pushShutdownSign();
			}

			listenerThread.interrupt();

		}catch(Exception e){
			//
			e.printStackTrace();
		}
	}

	class ListenerThread extends Thread {

		@Override
		public void run(){
			try{
				while(!isClosed){
					RedisQueueItem<T> value = takeFromHead();//cast exception? you should check.
					if (value.isShutdown()) {
						System.out.println(super.getName() + "shutdowning:" + isClosed);
						continue;
					}
					System.out.println("take...");
					//逐个执行
					if(value != null){
						try{
							System.out.println("value.item:" + value.getItem());
							listener.onMessage(value.getItem());
						}catch(Exception e){
							//
							e.printStackTrace();
							LoggerFactory.getLogger(this.getClass()).error("队列任务执行失败", e);
						}
					}
				}
			}catch(InterruptedException e){
				//
			}
		}


	}

}
